@php
use Illuminate\Support\Facades\DB;
use Illuminate\Support\Facades\Schema;

/**
 * Supplier Stock Report (custom, single file)
 * - Shows ONLY products tied to THIS supplier (contact) via purchases
 * - SKU: variations.sub_sku
 * - Stock: variation_location_details  (swap block if you use product_locations)
 * - Own table id to avoid conflict with system DataTables
 */

/* ---------- Figure out the supplier/contact id for this page ---------- */
$supplier_id =
    $contact->id
        ?? ($contact_details->id ?? null)
        ?? (request()->route('contact') ?? request()->route('id'))
        ?? request()->get('contact_id');

/* If we still couldn't detect it, you can harden this by pulling from the URL segment.
   Example (contacts/{id}): uncomment the line below if needed.
   $supplier_id = $supplier_id ?? (is_numeric(request()->segment(3)) ? (int)request()->segment(3) : null);
*/

$selected_location_id = request()->input('sr_location_id');

/* Business locations (fallback) */
if (!isset($business_locations) || empty($business_locations)) {
    try { $business_locations = DB::table('business_locations')->pluck('name', 'id')->toArray(); }
    catch (\Throwable $e) { $business_locations = []; }
}

/* Returns source (auto-detect) */
if (Schema::hasTable('sell_return_lines')) {
    $retSub = "
        SELECT srl.variation_id, t.location_id, SUM(srl.quantity) AS qty
        FROM sell_return_lines srl
        JOIN transactions t ON t.id = srl.transaction_id AND t.type='sell_return'
        GROUP BY srl.variation_id, t.location_id
    ";
} elseif (Schema::hasTable('transaction_sell_lines') && Schema::hasColumn('transaction_sell_lines','quantity_returned')) {
    $retSub = "
        SELECT tsl.variation_id, t.location_id, SUM(tsl.quantity_returned) AS qty
        FROM transaction_sell_lines tsl
        JOIN transactions t ON t.id = tsl.transaction_id AND t.type='sell'
        GROUP BY tsl.variation_id, t.location_id
    ";
} else {
    $retSub = "SELECT NULL AS variation_id, NULL AS location_id, 0 AS qty LIMIT 0";
}

/* Stock table (use product_locations block if needed) */
$stockSub = "
    SELECT vld.variation_id, vld.location_id, SUM(vld.qty_available) AS qty_available
    FROM variation_location_details vld
    GROUP BY vld.variation_id, vld.location_id
";
/*
$stockSub = "
    SELECT pl.variation_id, pl.location_id, SUM(pl.qty_available) AS qty_available
    FROM product_locations pl
    GROUP BY pl.variation_id, pl.location_id
";
*/

/* Variations that belong to THIS supplier via purchase transactions */
$supplierFilterJoin = '';
$supplierFilterWhere = '';
if (!empty($supplier_id)) {
    $supplier_id = (int)$supplier_id;
    $supplierFilterJoin = "
        LEFT JOIN (
            SELECT DISTINCT pl.variation_id
            FROM purchase_lines pl
            JOIN transactions t
              ON t.id = pl.transaction_id
             AND t.type = 'purchase'
             AND t.contact_id = {$supplier_id}
        ) sup ON sup.variation_id = v.id
    ";
    $supplierFilterWhere = " AND sup.variation_id IS NOT NULL ";
}

/* Final SQL (NO 'Total Unit Transferred' column) */
$sql = "
SELECT
    v.sub_sku                               AS sku,
    MIN(p.name)                             AS product_name,
    bl.name                                 AS location_name,
    COALESCE(v.dpp_inc_tax, v.default_purchase_price, 0) AS unit_price,
    COALESCE(st.qty_available, 0) AS current_stock,
    COALESCE(st.qty_available, 0) * COALESCE(v.dpp_inc_tax, v.default_purchase_price, 0) AS total_stock_price,
    COALESCE(pur.qty,  0) AS purchase_quantity,
    COALESCE(sld.qty,  0) AS total_sold,
    COALESCE(ret.qty,  0) AS total_returned,
    st.location_id
FROM variations v
JOIN products p ON p.id = v.product_id
{$supplierFilterJoin}
LEFT JOIN ( {$stockSub} ) st ON st.variation_id = v.id
LEFT JOIN business_locations bl ON bl.id = st.location_id
LEFT JOIN (
    SELECT pl.variation_id, t.location_id, SUM(pl.quantity) AS qty
    FROM purchase_lines pl
    JOIN transactions t ON t.id = pl.transaction_id AND t.type='purchase'
    GROUP BY pl.variation_id, t.location_id
) pur ON pur.variation_id = v.id AND pur.location_id = st.location_id
LEFT JOIN (
    SELECT tsl.variation_id, t.location_id, SUM(tsl.quantity) AS qty
    FROM transaction_sell_lines tsl
    JOIN transactions t ON t.id = tsl.transaction_id AND t.type='sell'
    GROUP BY tsl.variation_id, t.location_id
) sld ON sld.variation_id = v.id AND sld.location_id = st.location_id
LEFT JOIN ( {$retSub} ) ret ON ret.variation_id = v.id AND ret.location_id = st.location_id
WHERE v.sub_sku IS NOT NULL
{$supplierFilterWhere}
" . (!empty($selected_location_id) ? " AND st.location_id = ".(int)$selected_location_id : "") . "
GROUP BY v.sub_sku, st.location_id
ORDER BY v.sub_sku, bl.name, p.name
";

$rows = collect(DB::select(DB::raw($sql)));
@endphp

<div class="row">
	<div class="col-md-4">
	    <div class="form-group">
	        {!! Form::label('sr_location_id',  __('purchase.business_location') . ':') !!}
	        {!! Form::select('sr_location_id', $business_locations, $selected_location_id ?? null, [
                'class' => 'form-control select2',
                'style' => 'width:100%',
                'id'    => 'sr_location_id',
                'onchange' => 'this.form ? this.form.submit() : window.location = updateQueryString("sr_location_id", this.value);'
            ]) !!}
	    </div>
	</div>
</div>

@if(empty($supplier_id))
    <div class="alert alert-warning" role="alert">
        Couldn’t detect the supplier ID on this page. Results may be empty. If this persists, expose the contact id to this view (e.g. <code>$contact->id</code>).
    </div>
@endif

<div class="row">
	<div class="col-md-12">
		<div class="table-responsive">
            {{-- Custom id to avoid conflict with the system’s AJAX DataTable --}}
            <table class="table table-bordered table-striped" id="custom_sku_stock_table" width="100%">
                <thead>
                    <tr>
                        <th>@lang('sale.product')</th>
                        <th>@lang('product.sku')</th>
                        <th>@lang('purchase.purchase_quantity')</th>
                        <th>@lang('lang_v1.total_sold')</th>
                        <th>@lang('lang_v1.total_returned')</th>
                        <th>@lang('report.current_stock')</th>
                        <th>@lang('lang_v1.total_stock_price')</th>
                    </tr>
                </thead>
                <tbody>
                    @forelse($rows as $r)
                        <tr>
                            <td>
                                <div><strong>{{ $r->product_name }}</strong></div>
                                <div class="text-muted" style="font-size:12px;">{{ $r->location_name ?? '-' }}</div>
                            </td>
                            <td>{{ $r->sku }}</td>
                            <td class="text-right">{{ number_format((float)$r->purchase_quantity, 2) }}</td>
                            <td class="text-right">{{ number_format((float)$r->total_sold, 2) }}</td>
                            <td class="text-right">{{ number_format((float)$r->total_returned, 2) }}</td>
                            <td class="text-right">{{ number_format((float)$r->current_stock, 2) }}</td>
                            <td class="text-right">{{ number_format((float)$r->total_stock_price, 2) }}</td>
                        </tr>
                    @empty
                        <tr>
                            <td colspan="7" class="text-center text-muted">{{ __('lang_v1.no_data_available') }}</td>
                        </tr>
                    @endforelse
                </tbody>
            </table>
        </div>
	</div>
</div>

<script>
function updateQueryString(key, value) {
  const url = new URL(window.location.href);
  if (value === '' || value === null) url.searchParams.delete(key);
  else url.searchParams.set(key, value);
  return url.toString();
}

document.addEventListener('DOMContentLoaded', function () {
  var $tbl = $('#custom_sku_stock_table');
  if ($.fn.DataTable) {
    var opts = { paging: true, searching: true, ordering: true, pageLength: 25, order: [[1, 'asc']] };
    if ($.fn.dataTable.Buttons) { opts.dom = 'Bfrtip'; opts.buttons = ['csv', 'excel', 'pdf', 'print']; }
    $tbl.DataTable(opts);
  }
});
</script>
